/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics Hardware Configuration Information Manager */
/* Begun by Stig Telfer, API NetWorks, 22 February 2001 */

/* The concept of this module is to collect known fragments of information 
 * together, remembering something about where they were gathered from and 
 * what they are about.
 *
 * The collated data can then be presented in a uniform way which is intended
 * to be more helpful to the user.
 */

#include <stdarg.h>

#include "lib.h"
#include "uilib.h"
#include "smp.h"

#include "info.h"		/* We implement this interface */

/*------------------------------------------------------------------------*/
/* All known hardware configuration data is found here */

info_subject_t *info = NULL;


/*------------------------------------------------------------------------*/
/* Private stuff */

/* SMP lock for accessing the data structure */
static smp_mutex info_lock = M_FREE;


static const String src2str( const enum INFO_SOURCE src )
{
    static const struct { uint8 src; String str; } srcdata[] = {
	{ SRC_SROM,	"SROM" },
	{ SRC_DIAGS,	"Diags" },
	{ SRC_CHIPSET,	"Chipset" },
	{ SRC_I2C,	"I2C" },
	{ SRC_CPU, 	"Processor" },
	{ SRC_FIRMWARE,	"Firmware" },
	{ SRC_JUMPERS,	"Jumpers" },
	{ SRC_TIG,	"TIG bus" },
	{ SRC_MISC,	"Misc." }
    };

    int i;

    for( i=0; i<ARRAYLEN( srcdata ); i++ )
    {
	if ( srcdata[i].src == src )
	    return srcdata[i].str;
    }

    /* No match, something has gone wrong somewhere */
    return "ERROR";
}


static const String subj2str( const enum INFO_SUBJECT subj )
{
    static const struct { uint8 subj; String str; } subjdata[] = {
	{ SUBJ_CPU,		"Processor" },
	{ SUBJ_MEMORY,		"Memory" },
	{ SUBJ_FIRMWARE,	"Firmware" },
	{ SUBJ_CHIPSET,		"Chipset" },
	{ SUBJ_MOTHERBOARD,	"Motherboard" },
	{ SUBJ_MISC,		"Misc." }
    };

    int i;

    for( i=0; i<ARRAYLEN( subjdata ); i++ )
    {
	if ( subjdata[i].subj == subj )
	    return subjdata[i].str;
    }

    /* No match, something has gone wrong somewhere */
    return "ERROR";
}


static info_subject_t *lookup( const enum INFO_SUBJECT s )
{
    info_subject_t *S = info;

    while( S != NULL )
    {
	if( S->subject == s )
		break;

	S = S->next;
    }
    return S;				/* either the match or NULL */
}

static info_subject_t *new_S( const enum INFO_SUBJECT s )
{
    info_subject_t *tmp, *S = malloc( sizeof( info_subject_t ) );
    BUGCHECK( S == NULL );			/* Better not be... */

    S->subject = s;
    S->pearls = NULL;
    S->next = NULL;

    /* Put this new subject matter into the list of global knowledge */
    if ( info == NULL )
    {
	info = S;		/* global info list is empty */
    }
    else
    {
	tmp = info;
	while( tmp->next != NULL )
	    tmp = tmp->next;

	tmp->next = S;		/* added to the end of the subject list */
    }

    return S;
}


static info_pearl_t *new_P( const enum INFO_SOURCE src, const String Attribute,
		const String Value )
{
    info_pearl_t *P = malloc( sizeof( info_pearl_t ) );
    BUGCHECK( P == NULL	);			/* Better not be... */

    P->source = src;

    P->attribute = malloc( strlen( Attribute ) + 1 );
    BUGCHECK( P->attribute == NULL );
    strcpy( P->attribute, Attribute );

    P->value = malloc( strlen( Value ) + 1 );
    BUGCHECK( P->value == NULL );
    strcpy( P->value, Value );

    P->next = NULL;

    return P;
}


static BOOLEAN pearl_match( const info_pearl_t *P, const enum INFO_SOURCE src,
		const String A, const String V )
{
    if ( P->source != src )
	return FALSE;

    if ( strcmp( P->attribute, A ) != 0 )
	return FALSE;

    if ( strcmp( P->value, V ) != 0 )
	return FALSE;

    return TRUE;			/* They are the same after all! */
}


/*------------------------------------------------------------------------*/
/* Access functions */

/* Add a new piece of information to what is known */
/* Note: in an effort to avoid duplication of data, we simply discard
 * replicated info.
 */
#define MAXVALUELEN	64

void info_submit( const enum INFO_SOURCE src, const enum INFO_SUBJECT subj,
		  const String Attribute, const String Valuefmt, ... )
{
    info_subject_t *S;
    info_pearl_t *P;
    char valbuf[ MAXVALUELEN ];
    va_list ap;
    va_start( ap, Valuefmt );
    vsprintf_dbm( valbuf, Valuefmt, ap );
    va_end( ap );

    smp_acquire( &info_lock );	/* -------- Begin mutual exclusion -------- */

    S = lookup( subj );
    if ( S == NULL )
	S = new_S( subj );

    if( S->pearls == NULL )
    { 
	S->pearls = new_P( src, Attribute, valbuf );
    }
    else
    {
	P = S->pearls;

	if ( pearl_match( P, src, Attribute, valbuf ) )
		return;

	while( P->next != NULL )
	{
		if ( pearl_match( P->next, src, Attribute, valbuf ) )
			return;

		P = P->next;
	}

	P->next = new_P( src, Attribute, valbuf );
    }

    smp_release( &info_lock );	/* -------- End mutual exclusion -------- */
}


/* Dump all known information to the log stream */
void info_log( void )
{
    info_subject_t *S;
    info_pearl_t *P;

    mobo_logf( LOG_INFO "Information about this system\n"
	       LOG_INFO "-----------------------------\n" );

    smp_acquire( &info_lock );	/* -------- Begin mutual exclusion -------- */
    S = info;
    while( S != NULL )
    {
	mobo_logf( LOG_INFO "%s information:\n", subj2str( S->subject ) );

	P = S->pearls;
	while( P != NULL )
	{
	    mobo_logf( LOG_INFO "  %-30s : %-30s [%s]\n",
			P->attribute, P->value, src2str( P->source ) );
	    P = P->next;
	}

	S = S->next;
    }
    smp_release( &info_lock );	/* -------- End mutual exclusion -------- */
}


/* List all known information on the screen */

#define NROWS		16		/* number of displayable rows */
#define INFO_PROMPT	"Info> "

static void subject_dump( const info_subject_t *S )
{
    info_pearl_t *P;
    int lines;

    P = S->pearls;

    while( P != NULL )
    {
	mobo_box( r_scrn, "Information known" );
	printf_dbm( "Subject: %s\r\r", subj2str( S->subject ) );

	for( lines=0; lines < NROWS && P != NULL; lines++, P=P->next )
	{
	    printf_dbm( " %-30s : %-30s [%s]\r", 
			P->attribute, P->value, src2str( P->source ) );
	}

	mobo_goto( p_help );
	printf_dbm( "Press any key to continue..." );
	mobo_key( 0 );
    }
}

#define INPUTBUFLEN	4

DBM_STATUS info_dump( int argc, char *argv[] )
{
    info_subject_t *S;
    int i;
    int rval;
    char buf[ INPUTBUFLEN ];
    const Point list = { 2, 3 };


    do {
	/* Draw main user interface */
	mobo_box( r_scrn, "Information about this system" );

	mobo_goto( list );
	printf_dbm( "Hardware information categories:\r\r" );
	for( i=SUBJ_FIRST; i<=SUBJ_LAST; i++ )
	{
	    printf_dbm( "%2d) %s information\r", i, subj2str( i ) );
	}

	mobo_goto( p_help );
	printf_dbm( "Please select a category by number to examine "
			"(or RETURN to quit):" );

	mobo_goto( p_prompt );
	printf_dbm( INFO_PROMPT );

        rval = mobo_input( buf, INPUTBUFLEN );

	/* Did the user hit return to quit? */
        if( rval == 0 )
                break;

	/* Malformed input? */
	if( sscanf( buf, "%d", &rval ) != 1 )
		continue;

	/* Does it relate to a valid choice? */
        if ( rval < SUBJ_FIRST || rval > SUBJ_LAST )
                continue;

	/* Find the subject in question */
	smp_acquire( &info_lock );	/* ---- Begin mutual exclusion ---- */
	S = lookup( rval );
	if( S == NULL )
	{
	    mobo_alertf( "No information",
		"%s: No information is currenctly recorded about that subject",
		subj2str( rval ) );
	    continue;
	}
	else
	{
	    subject_dump( S );
	}
	smp_release( &info_lock );	/* ---- End mutual exclusion ---- */

    } while ( TRUE );

    return STATUS_SUCCESS;
}

